<?php
/**
 * Plugin Name:       Pretty Links Alternative (Stable)
 * Description:       A professional, fully functional link tracking and redirection plugin. All features work correctly.
 * Version:           6.0 (Stable & Self-Contained)
 * Author:            Digital Marketing Support
 * Author URI:        https://digitalmarketingsupport.org/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       pla
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

// ===================================================================================
// 1. POST TYPE REGISTRATION AND ADMIN COLUMNS
// ===================================================================================

add_action('init', 'pla_register_post_type');
add_filter('manage_pretty-link_posts_columns', 'pla_admin_columns');
add_action('manage_pretty-link_posts_custom_column', 'pla_custom_column_content', 10, 2);

function pla_register_post_type() {
    $labels = [
        'name' => _x('Pretty Links', 'Post type general name', 'pla'),
        'singular_name' => _x('Pretty Link', 'Post type singular name', 'pla'),
        'menu_name' => _x('Pretty Links Alt', 'Admin Menu text', 'pla'),
        'add_new' => __('Add Link', 'pla'),
        'add_new_item' => __('Add New Link', 'pla'),
        'edit_item' => __('Edit Link', 'pla'),
        'all_items' => __('All Links', 'pla'),
    ];
    $args = [
        'labels' => $labels,
        'public' => false,
        'publicly_queryable' => true,
        'show_ui' => true,
        'show_in_menu' => true,
        'rewrite' => ['slug' => get_option('pla_base_slug', 'go'), 'with_front' => false],
        'capability_type' => 'post',
        'hierarchical' => false,
        'menu_position' => 20,
        'menu_icon' => 'dashicons-admin-links',
        'supports' => ['title'],
    ];
    register_post_type('pretty-link', $args);
}

function pla_admin_columns($columns) {
    return [
        'cb' => $columns['cb'],
        'title' => __('Link Title', 'pla'),
        'pretty_link' => __('Pretty Link', 'pla'),
        'clicks' => __('Clicks', 'pla'),
        'target_url' => __('Target URL', 'pla'),
        'date' => $columns['date'],
    ];
}

function pla_custom_column_content($column, $post_id) {
    switch ($column) {
        case 'pretty_link':
            $post = get_post($post_id);
            $pretty_link = rtrim(get_site_url(), '/') . '/' . get_option('pla_base_slug', 'go') . '/' . $post->post_name;
            echo '<a href="' . esc_url($pretty_link) . '" target="_blank">' . esc_url(urldecode($pretty_link)) . '</a><input type="text" value="' . esc_attr($pretty_link) . '" readonly onfocus="this.select();" style="width:100%; margin-top:4px;"/>';
            break;
        case 'target_url':
            echo esc_url(get_post_meta($post_id, '_pla_target_url', true));
            break;
        case 'clicks':
            $click_count = get_post_meta($post_id, '_pla_click_count', true) ?: '0';
            echo '<div class="pla-clicks-column"><span class="pla-click-count-number" id="pla-click-count-' . $post_id . '">' . absint($click_count) . '</span><a href="#" class="pla-reset-click" data-postid="' . $post_id . '" data-nonce="' . wp_create_nonce('pla_reset_nonce_' . $post_id) . '" title="Reset click count">Reset</a></div>';
            break;
    }
}

// ===================================================================================
// 2. METABOXES AND THE FULLY WORKING SAVE FUNCTION
// ===================================================================================

add_action('add_meta_boxes', 'pla_add_metaboxes');
add_action('save_post_pretty-link', 'pla_save_link_data', 10, 2);

function pla_add_metaboxes() {
    add_meta_box('pla_link_settings', 'Link Settings', 'pla_link_settings_cb', 'pretty-link', 'normal', 'high');
}

function pla_save_link_data($post_id, $post) {
    // --- Security and Validation ---
    if (!isset($_POST['pla_link_nonce']) || !wp_verify_nonce($_POST['pla_link_nonce'], 'pla_save_link_data')) return;
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    // --- Save Slug ---
    if (isset($_POST['post_name'])) {
        $new_slug = sanitize_title($_POST['post_name']);
        if ($post->post_name !== $new_slug) {
            remove_action('save_post_pretty-link', 'pla_save_link_data', 10);
            wp_update_post(['ID' => $post_id, 'post_name' => $new_slug]);
            add_action('save_post_pretty-link', 'pla_save_link_data', 10, 2);
        }
    }
    
    // --- Save Basic Text/Select Fields ---
    $fields = ['pla_target_url', 'pla_redirect_type', 'pla_notes'];
    foreach ($fields as $field) {
        if (isset($_POST[$field])) {
            update_post_meta($post_id, '_' . $field, sanitize_text_field($_POST[$field]));
        }
    }
    
    // --- THIS IS THE FIX: Save Advanced Tab Checkboxes ---
    $checkboxes = ['pla_nofollow', 'pla_sponsored', 'pla_param_forwarding', 'pla_tracking'];
    foreach ($checkboxes as $checkbox) {
        update_post_meta($post_id, '_' . $checkbox, isset($_POST[$checkbox]) ? '1' : '0');
    }
}

function pla_link_settings_cb($post) {
    wp_nonce_field('pla_save_link_data', 'pla_link_nonce');
    ?>
    <div class="pla-metabox-wrapper">
        <nav class="nav-tab-wrapper"><a href="#pla-tab-basic" class="nav-tab nav-tab-active">Basic</a><a href="#pla-tab-advanced" class="nav-tab">Advanced</a></nav>
        <div id="pla-tab-basic" class="pla-tab-content active"><?php pla_render_basic_tab($post); ?></div>
        <div id="pla-tab-advanced" class="pla-tab-content"><?php pla_render_advanced_tab($post); ?></div>
    </div>
    <?php
}

function pla_render_basic_tab($post) {
    $target_url = get_post_meta($post->ID, '_pla_target_url', true);
    $redirect_type = get_post_meta($post->ID, '_pla_redirect_type', true) ?: '307';
    $base_slug = get_option('pla_base_slug', 'go');
    ?>
    <table class="form-table">
        <tr>
            <th><label for="pla_target_url">Target URL</label></th>
            <td><input type="url" id="pla_target_url" name="pla_target_url" value="<?php echo esc_url($target_url); ?>" class="large-text" required /></td>
        </tr>
        <tr>
            <th><label for="post_name">Pretty Link Slug</label></th>
            <td>
                <span class="pla-base-url"><?php echo esc_url(get_site_url() . '/' . $base_slug . '/'); ?></span>
                <input type="text" name="post_name" id="post_name" value="<?php echo esc_attr($post->post_name); ?>" />
            </td>
        </tr>
        <tr>
            <th><label for="pla_redirect_type">Redirection</label></th>
            <td><select name="pla_redirect_type" id="pla_redirect_type"><option value="307" <?php selected($redirect_type, '307'); ?>>307 (Temporary)</option><option value="302" <?php selected($redirect_type, '302'); ?>>302 (Temporary)</option><option value="301" <?php selected($redirect_type, '301'); ?>>301 (Permanent)</option></select></td>
        </tr>
    </table>
    <?php
}

function pla_render_advanced_tab($post) {
    $is_new_post = get_current_screen()->action === 'add';
    $nofollow = get_post_meta($post->ID, '_pla_nofollow', true);
    $sponsored = get_post_meta($post->ID, '_pla_sponsored', true);
    $param_forwarding = get_post_meta($post->ID, '_pla_param_forwarding', true);
    $tracking = get_post_meta($post->ID, '_pla_tracking', true);
    ?>
    <table class="form-table">
        <tr>
            <th>Link Options</th>
            <td>
                <p><label><input type="checkbox" name="pla_nofollow" value="1" <?php checked($is_new_post || $nofollow === '1'); ?>> No Follow</label></p>
                <p><label><input type="checkbox" name="pla_sponsored" value="1" <?php checked($sponsored === '1'); ?>> Sponsored</label></p>
            </td>
        </tr>
        <tr>
            <th>Parameter Forwarding</th>
            <td><label><input type="checkbox" name="pla_param_forwarding" value="1" <?php checked($param_forwarding === '1'); ?>> Enable</label></td>
        </tr>
        <tr>
            <th>Tracking</th>
            <td><label><input type="checkbox" name="pla_tracking" value="1" <?php checked($is_new_post || $tracking === '1'); ?>> Enable Click Tracking</label></td>
        </tr>
    </table>
    <?php
}

// ===================================================================================
// 3. REDIRECT HANDLING
// ===================================================================================

add_action('init', 'pla_redirect_handler', 1);

function pla_redirect_handler() {
    if (is_admin()) return;
    
    $request_path = trim(strtolower(strtok($_SERVER['REQUEST_URI'], '?')), '/');
    if (empty($request_path)) return;
    
    $base_slug = get_option('pla_base_slug', 'go');

    if (strpos($request_path, $base_slug . '/') === 0) {
        $link_slug = basename($request_path);
        if (!$link_slug) return;
        
        $link_query = new WP_Query([
            'name'           => $link_slug,
            'post_type'      => 'pretty-link',
            'post_status'    => 'publish',
            'posts_per_page' => 1,
            'fields'         => 'ids',
        ]);
        
        if ($link_query->have_posts()) {
            $post_id = $link_query->posts[0];
            $target_url = get_post_meta($post_id, '_pla_target_url', true);
            if (empty($target_url)) return;

            if (get_post_meta($post_id, '_pla_tracking', true) == '1') {
                $click_count = (int)get_post_meta($post_id, '_pla_click_count', true);
                update_post_meta($post_id, '_pla_click_count', $click_count + 1);
            }

            if (get_post_meta($post_id, '_pla_param_forwarding', true) == '1' && !empty($_SERVER['QUERY_STRING'])) {
                $target_url .= (strpos($target_url, '?') === false ? '?' : '&') . $_SERVER['QUERY_STRING'];
            }
            
            $redirect_type = (int)get_post_meta($post_id, '_pla_redirect_type', true) ?: 307;
            wp_redirect(esc_url_raw($target_url), $redirect_type);
            exit;
        }
    }
}

// ===================================================================================
// 4. EMBEDDED CSS AND JAVASCRIPT (No external files needed)
// ===================================================================================

add_action('admin_head', 'pla_embed_admin_css');
add_action('admin_footer', 'pla_embed_admin_js');

function pla_embed_admin_css() {
    $screen = get_current_screen();
    if ('pretty-link' != $screen->post_type) return;
    ?>
    <style type="text/css">
        .pla-clicks-column { display: flex; align-items: center; }
        .pla-clicks-column .pla-reset-click { margin-left: 12px; font-size: 11px; color: #a00; cursor: pointer; }
        .pla-clicks-column .pla-reset-click:hover { color: #f00; }
        .pla-metabox-wrapper .pla-tab-content { display: none; padding: 20px 0 10px 0; }
        .pla-metabox-wrapper .pla-tab-content.active { display: block; }
        .pla-base-url { padding: 5px; background: #f0f0f1; border: 1px solid #ddd; border-right: 0; display: inline-block; vertical-align: middle; }
        #post_name { vertical-align: middle; }
    </style>
    <?php
}

function pla_embed_admin_js() {
    $screen = get_current_screen();
    if ('pretty-link' != $screen->post_type) return;
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        'use strict';
        
        // Tab switching
        $('.pla-metabox-wrapper .nav-tab').on('click', function(e) {
            e.preventDefault();
            var target = $(this).attr('href');
            $(this).addClass('nav-tab-active').siblings().removeClass('nav-tab-active');
            $(target).addClass('active').siblings('.pla-tab-content').removeClass('active');
        });

        // AJAX Click Reset - This is the fully working version
        $('body').on('click', '.pla-reset-click', function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to reset the click count to zero?')) {
                return;
            }

            var $button = $(this);
            var postId = $button.data('postid');
            var nonce = $button.data('nonce');
            var $clickCountSpan = $('#pla-click-count-' + postId);
            
            var $spinner = $('<span class="spinner is-active" style="display:inline-block; float:none; vertical-align:middle; margin-left:12px;"></span>');
            
            $button.hide();
            $button.after($spinner);

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'pla_reset_clicks',
                    post_id: postId,
                    nonce: nonce
                },
                success: function(response) {
                    if (response.success) {
                        $clickCountSpan.text('0').css('color', 'green');
                        setTimeout(function() { $clickCountSpan.css('color', ''); }, 2000);
                    } else {
                        alert('Failed: ' + response.data.message);
                    }
                },
                error: function() {
                    alert('An error occurred. Please try again.');
                },
                complete: function() {
                    $spinner.remove();
                    $button.show();
                }
            });
        });
    });
    </script>
    <?php
}

// ===================================================================================
// 5. AJAX HANDLER
// ===================================================================================
add_action('wp_ajax_pla_reset_clicks', 'pla_ajax_reset_clicks_handler');

function pla_ajax_reset_clicks_handler() {
    $post_id = isset($_POST['post_id']) ? (int)$_POST['post_id'] : 0;
    if (!$post_id || !check_ajax_referer('pla_reset_nonce_' . $post_id, 'nonce', false)) {
        wp_send_json_error(['message' => 'Security check failed.']);
    }
    if (!current_user_can('edit_post', $post_id)) {
        wp_send_json_error(['message' => 'Permission denied.']);
    }
    
    update_post_meta($post_id, '_pla_click_count', 0);
    wp_send_json_success(['new_count' => 0]);
}

// ===================================================================================
// 6. PLUGIN ACTIVATION HOOK
// ===================================================================================
register_activation_hook(__FILE__, function() {
    pla_register_post_type();
    add_option('pla_base_slug', 'go');
    flush_rewrite_rules();
});